<?php
/**
 * Linguise
 *
 * We developed this code with our hearts and passion.
 * We hope you found it useful, easy to understand and to customize.
 * Otherwise, please feel free to contact us at contact@linguise.com
 *
 * @package   Linguise
 * @copyright Copyright (C) 2021 Linguise (http://www.linguise.com). All rights reserved.
 * @license   GNU General Public License version 2 or later; http://www.gnu.org/licenses/gpl-2.0.html
 */

defined('_JEXEC') or die;

class PlgSystemLinguise extends JPlugin
{
    /**
     * Hook begore anything happen to translate this request if we have to
     *
     * @param $subject
     * @param array $config
     */
    public function __construct(&$subject, $config = array())
    {
        if (JFactory::getApplication()->isClient('administrator')) {
            return parent::__construct($subject, $config);
        }

        JLoader::register('LinguiseHelper', JPATH_ADMINISTRATOR . '/components/com_linguise/helpers/linguise.php');

        if (!LinguiseHelper::getOption('linguise_field_token')) {
            return parent::__construct($subject, $config);
        }

        if (!empty(LinguiseHelper::getOption('joomla_languages'))) {
            // We have a joomla multilingual, we need to unset the session to avoid Joomla language redirection
            JFactory::getSession()->set('plg_system_languagefilter.language', JFactory::getLanguage()->getDefault());
        }

        include_once('vendor' . DIRECTORY_SEPARATOR . 'autoload.php');

        $base_dir = JUri::root(true);
        $path = substr($_SERVER['REQUEST_URI'], strlen($base_dir));

        $path = parse_url('https://localhost/' . ltrim($path, '/'), PHP_URL_PATH);

        $parts = explode('/', trim($path, '/'));

        if (!count($parts)) {
            return parent::__construct($subject, $config);
        }

        if ($parts[0] === 'index.php') {
            array_splice($parts, 0, 1);
        }

        if (!count($parts) || $parts[0] === '') {
            return parent::__construct($subject, $config);
        }

        $language = $parts[0];

        if (!empty($_SERVER['HTTP_LINGUISE_ORIGINAL_LANGUAGE']) && $language !== LinguiseHelper::getOption('language_default') && in_array($language, LinguiseHelper::getOption('joomla_languages'))) {
            throw new Exception(JText::_('JERROR_LAYOUT_PAGE_NOT_FOUND'), 404);
        }

        if (!in_array($language, array_merge(LinguiseHelper::getOption('languages_enabled'), array('zz-zz'))) || in_array($language, LinguiseHelper::getOption('joomla_languages'))) {
            return parent::__construct($subject, $config);
        }

        $_GET['linguise_language'] = $language;

        include_once('script.php');
    }


    /**
     * Languages have been initialized, we can fill the current language option
     *
     * @return void
     */
    public function onAfterRoute() {

        if (!JFactory::getApplication()->isClient('site')) {
            return;
        }

        JLoader::register('LinguiseHelper', JPATH_ADMINISTRATOR . '/components/com_linguise/helpers/linguise.php');

        $headers = getallheaders();
        $joomla_current_language = LinguiseHelper::getOption('joomla_default_language');
        if (!empty($joomla_current_language)) {
            $current_language = substr(JFactory::getLanguage()->getTag(), 0, 2);
        } else {
            $current_language = LinguiseHelper::getOption('language_default');
        }

        $header_language = null;
        if (!empty($headers['Linguise-Original-Language']) && preg_match('/[a-z]{2}(-[a-z]{2})?/', $headers['Linguise-Original-Language'])) {
            $header_language = $headers['Linguise-Original-Language'];
        }
        LinguiseHelper::setOption('current_language', $header_language ?: $current_language);

        LinguiseHelper::browserRedirect();
    }

    static private $_before_rendered = false;

    /**
     * Before Render
     *
     * @return void
     * @throws \Exception Throw when application can not start
     * @since  version
     */
    public function onBeforeLinguiseRender()
    {
        if (self::$_before_rendered) {
            return;
        }
        self::$_before_rendered = true;

        $app = JFactory::getApplication();
        // get the router
        if (!$app->isClient('site')) {
            return;
        }

        JLoader::register('ModLanguagesHelper', JPATH_ROOT . '/modules/mod_languages/helper.php');

        $options = LinguiseHelper::getOptions();

        $uri = JUri::getInstance();

        $languages_content = file_get_contents(JPATH_ROOT. '/modules/mod_linguise/assets/languages.json');
        $languages_names = json_decode($languages_content);

        // Get from module parameters the enable languages
        $languages_enabled_param = LinguiseHelper::getOption('languages_enabled', []);
        // Get the default language
        $default_language = LinguiseHelper::getOption('language_default', 'en');
        $language_name_display = LinguiseHelper::getOption('language_name_display', 'en');

        // Generate language list with default language as first item
        if ($language_name_display === 'en') {
            $language_list = array($default_language => $languages_names->{$default_language}->name);
        } else {
            $language_list = array($default_language => $languages_names->{$default_language}->original_name);
        }

        foreach ($languages_enabled_param as $language) {
            if ($language === $default_language) {
                continue;
            }

            if (!isset($languages_names->{$language})) {
                continue;
            }

            if ($language_name_display === 'en') {
                $language_list[$language] = $languages_names->{$language}->name;
            } else {
                $language_list[$language] = $languages_names->{$language}->original_name;
            }
        }

        if (preg_match('@(\/+)$@', parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH), $matches) && !empty($matches[1])) {
            $trailing_slashes = $matches[1];
        } else {
            $trailing_slashes = '';
        }

        $base = rtrim($uri->base(true),'/');
        LinguiseHelper::setOption('languages', $language_list);
        LinguiseHelper::setOption('base', $base);
        $original_path = rtrim(substr(rtrim(parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH), '/'), strlen($base)), '/');
        LinguiseHelper::setOption('original_path', $original_path);
        LinguiseHelper::setOption('trailing_slashes', $trailing_slashes);

        if ($options['alternate_link']) {

            $joomla_languages = [];
            if (!empty($options['joomla_languages'])) {
                $params = new \Joomla\Registry\Registry();
                $joomla_language_list = ModLanguagesHelper::getList($params);
                foreach ($joomla_language_list as $joomla_language) {
                    $jlang = substr($joomla_language->lang_code, 0, 2);
                    if (!in_array($jlang, $options['joomla_languages'])) {
                        continue;
                    }

                    if ($jlang === $options['joomla_default_language']) {
                        // Remove the language tag from the link
                        $link = preg_replace('/^' . str_replace('/', '\/', $base) . '\/' . $jlang . '/', '', $joomla_language->link);
                    } else {
                        $link = $joomla_language->link;
                    }

                    $joomla_languages[$jlang] = $link;
                }
            }

            $scheme = $uri->getScheme();
            $host = $uri->getHost();
            $path = $original_path;
            $query = $uri->getQuery();
            $alternates = $language_list;
            $alternates['x-default'] = 'x-default';

            if (!empty($joomla_languages) && $options['current_language'] !== $options['language_default'] && in_array($options['current_language'], array_keys($joomla_languages))) {
                // We are in a Joomla managed language but not the default one. Let's replace the path by the untranslated path we get from Joomla
                $path = $joomla_languages[$options['language_default']];
            }

            $dbo = JFactory::getDbo();
            foreach ($alternates as $language_code => $language) {

                if (!empty($joomla_languages[$language_code]) && $language_code !== $options['joomla_default_language']) {
                    // This language is managed by joomla, get the url from joomla
                    $url = $scheme . '://' . $host . $joomla_languages[$language_code];
                } else {

                    if ($language_code === $options['joomla_default_language']) {
                        // This is the default Joomla language
                        $url = $scheme . '://' . $host . $base . $path . (!empty($query) ? '?' . $query : '');
                    } else {
                        $db_query = 'SELECT * FROM #__linguise_urls WHERE hash_source=' . $dbo->quote(md5($path)) . ' AND language=' . $dbo->quote($language_code);
                        try {
                            // Wrap in a try catch in case databse has not been already created by linguise script
                            $url_translation = $dbo->setQuery($db_query)->loadAssoc();
                        } catch (\Exception $e) {}


                        if (!empty($url_translation)) {
                            $url = $scheme . '://' . $host . $base  . htmlentities($url_translation['translation']) . (!empty($query) ? '?' . $query : '');
                        } else {
                            $url = $scheme . '://' . $host . $base . (in_array($language_code, array($default_language, 'x-default')) ? '' : '/' . $language_code) . $path . $trailing_slashes . (!empty($query) ? '?' . $query : '');
                        }
                    }
                }

                JFactory::getDocument()->addCustomTag(
                    '<link 
                    rel="alternate" 
                    hreflang="' . $language_code . '" 
                    href="' . $url . '" 
                />'
                );
            }

        }

    }

    public function onAfterRenderModule($module, $attribs)
    {
        if (JFactory::getApplication()->isClient('administrator')) {
            return;
        }

        if ($module->module !== 'mod_menu') {
            return;
        }

        $params = new \Joomla\Registry\Registry($module->params);
        if (version_compare(JVERSION, '4.0.0') >= 0) {
            $list = Joomla\Module\Menu\Site\Helper\MenuHelper::getList($params);
        } else {
            JLoader::register('ModMenuHelper', JPATH_BASE . '/modules/mod_menu/helper.php');
            $list = ModMenuHelper::getList($params);
        }



        foreach ($list as $item) {
            if ($item->type === 'component' && $item->component === 'com_linguise') {
                JFactory::getApplication()->triggerEvent('onBeforeLinguiseRender');

                JLoader::register('LinguiseHelper', JPATH_ADMINISTRATOR . '/components/com_linguise/helpers/linguise.php');

                LinguiseHelper::addAssets();
            }
        }
    }

    public function onExtensionAfterSave($context, $table, $isNew)
    {
        if ($context !== 'com_modules.module' || $table->module !== 'mod_linguise') {
            return;
        }
    }

    public function onContentAfterSave($context, $table, $isNew)
    {
        if ($context !== 'com_advancedmodules.module' || $table->module !== 'mod_linguise') {
            return;
        }
    }

    /**
     * Add the right default class to the menu anchor attribute
     *
     * @param $form
     * @param $data
     * @return boolean
     */
    public function onContentPrepareForm($form, $data) {
        if (JFactory::getApplication()->isClient('site')) {
            return true;
        }

        if ((!empty($data->params['option']) && is_array($data->params) && ($data->params['option'] !== 'com_linguise' || $data->params['view'] !== 'linguise')) ||
            $form->getName() !== 'com_menus.item' || !$form->getField('menu-anchor_css', 'params')) {
            return true;
        }

        $form->setFieldAttribute('menu-anchor_css', 'default', 'joomla_linguise_switcher', 'params');
        return true;
    }

    /**
     * Make sure we have the right class in the menu item after form submission
     *
     * @param $context
     * @param $item
     * @param $isNew
     * @return boolean
     * @throws Exception
     */
    public function onContentBeforeSave($context, $item, $isNew) {
        if (JFactory::getApplication()->isClient('site')) {
            return true;
        }

        if ($context !== 'com_menus.item'  || JFactory::getApplication()->isClient('site') || $item->getTableName() !== '#__menu') {
            return true;
        }

        $params = json_decode($item->get('params'));
        if (!$params) {
            return true;
        }

        if ($item->get('link') !== 'index.php?option=com_linguise&view=linguise') {
            if (preg_match('/(^| )joomla_linguise_switcher($| )/', $params->{'menu-anchor_css'})) {
                $params->{'menu-anchor_css'} = str_replace('joomla_linguise_switcher', '', $params->{'menu-anchor_css'});
                $item->set('params', json_encode($params));
                return true;
            }
            return true;
        }

        // Check if we have the right class
        if (preg_match('/(^| )joomla_linguise_switcher($| )/', $params->{'menu-anchor_css'}) && preg_match('/(^| )noindex, nofollow($| )/', $params->{'robots'})) {
            return true;
        }

        if ($params->{'menu-anchor_css'} !== '') {
            $params->{'menu-anchor_css'} .= ' ';
        }

        $params->{'menu-anchor_css'} = str_replace('joomla_linguise_switcher', '', $params->{'menu-anchor_css'});
        $params->{'menu-anchor_css'}.= 'joomla_linguise_switcher';
        $params->{'robots'} = 'noindex, nofollow';
        $item->set('params', json_encode($params));

        return true;
    }
}